/*

  Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
  Released Under a Simplified BSD (FreeBSD) License
  for academic, personal & non-commercial use.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  The views and conclusions contained in the software and documentation are those
  of the author and should not be interpreted as representing official policies,
  either expressed or implied, of the FreeBSD Project.

  A Commercial License is also available for commercial use with
  special restrictions and obligations at a one-off fee. See links at:
  1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
  2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
  Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

  History of the GPU Framework & Rendering Engine project:
  =======================================================
  Version  1.0: Main core framework implemented along with the VBOs tests using Visual Studio 2010.
  Version  2.0: The Super Quadrics tests were implemented.
  Version  3.0: The Model Loader tests were implemented.
  Version  4.0: The Stencil Buffer outline implemented in both the Super Quadrics & Model Loader tests.
  Version  5.0: Tessellation Shaders implemented in both the Super Quadrics & Model Loader tests.
  Version  6.0: Various shader optimizations, latest Glew/FreeGlut & Boost version 1.53 updates (maintenance release).
  Version  7.0: OpenGLRenderingEngine namespace & OOP OpenGL Shader Loader Library added.
  Version  8.0: The Particle Rendering Super Quadrics & Particle Rendering Model Loader tests were implemented. Sphere raytraced perspective-correct Impostor implemented with optional double precision raytracing.
  Version  9.0: The Depth-Of-Field & Order-Independent-Transparency (O-I-T) tests with Deferred Shading & Fresnel term were implemented.
                Codebase refactored to be fully C++11 compliant using Visual Studio 2013. Relevant Boost dependencies were removed.
  Version 10.0: The merged G-Buffer effects were implemented: Depth-Of-Field, Edge Ehancement, Hemispherical Screen-Space Ambient Occlusion (HSSAO) & HSSAO Plus with Sharpness Filter.
                Many other visualization related bugs were ironed out & the O-I-T Deferred Shading was further optimized.
  Version 11.0: Cylinder raytraced perspective-correct Impostor implemented with optional double precision raytracing. Sphere Impostors using geometry shaders further optimized.
                A better GPU-side GLSL randomizer implementation was also added & used from the GLSL shader framework. A wireframe Impostor-based rendering mode was implemented for Impostor-based tests.
  Version 11.5: Further advancements of codebase to more C++11 compliance, also updating all relevant used libraries (freeglut version 3.0.0, glew version 1.12.0). Added a fast procedural quad texture creation technique through dummy VAOs and vertex shader usage.
                Added a fast jitter environment mapping technique used as material in 3D objects. Added a more accurate programmable shader interpolation support for sphere impostors as well as early radius check exit.
                Added a programmable cube capping technique in test 1 also using the stencil buffer. Added GL debug context usage.
  Version 12.0: N-CP support was further advanced with a proper wrapper (parallelFor(lambda func) PPL-style) and unit tests.
                Advanced the FBO class depth/stencil support & fixed some long-standing bugs. Added multi sample FBO support for FXAA FBO effect giving even better antialias results to all tests.
                Used an RGBA format (instead of the RGB one) for the FBO used for the G-Buffer effects to avoid an Intel GPU driver crash with the rendering engine. Still errors to fix there though.
                Verified compilation compliance with C++14 using Visual Studio 2015 with all relevant upgrades & changes.
  Version 13.0: Total overhaul to the ShaderFilesHeaderGenerator system, resulting to faster compilations with smaller final executables & generally smaller memory footprint. A config file was also added.
                Parallelized the build system in VS2015 with much faster build times as a result. Smoother & bicubic texure lookup filters added to the shader codebase. Glew 2.0 added with new GL4.5 extensions.
  Version 14.0: Total overhaul to the build system, compartmentalizing, creating separate dynamically linked libraries, unit tests & documentation for each module/subsystem of the OpenGL Rendering Engine. NVidia CUDA support was also added along with an HG repository. More detail below:
                1. CMake 3.9+ build system automation
                2. GoogleTest 1.8+ verification framework
                3. Doxygen 1.8+ with LaTeX pdf support in-source documentation
                4. CUDA 9.2+/NVCC component built-in with C++11/14 compliance
                5. HG/Mercurial 4.6+ repository

*/

/**
*
*  GPUFrameworkMain.cpp:
*  ====================
*  GPUFrameworkMain holds the application's void main() function and the core Factory Design Pattern of which GLUT test to load.
*
* @author Thanos Theo, 2009-2018
* @version 14.0.0.0
*/

#include "ConfigFile.h"
#include "UtilityFunctions.h"
#include "SIMDVectorizations.h"
#include "TestGLUTInterface.h"
#include "TestAbstractBase.h"
#include "CubeCappingTest.h"
#include "SuperQuadricsTest.h"
#include "ModelLoaderTest.h"
#include "ParticleRenderingSuperQuadricsTest.h"
#include "ParticleRenderingModelLoaderTest.h"
#include "ModelLoaderDepthOfFieldTest.h"
#include "ModelLoaderOITTest.h"
#include <exception>
#include <string>
#include <array>
#include <cstdio>
#include <memory>

using namespace std;
using namespace OpenGLRenderingEngineTests;
using namespace Utils::UtilityFunctions;

#ifdef _WIN32 // Windows system specific
  // Applications exporting this symbol with this value will be automatically
  // directed to the high-performance GPU on nVidia Optimus systems
  extern "C"
  {
    _declspec(dllexport) DWORD NvOptimusEnablement = 0x00000001;
  }
#endif

namespace // anonymous namespace used instead of deprecated 'static' keyword used for cpp variable locality
{
  const float  GLUT_WINDOW_TO_SCREEN_SIZE_RATIO                = 0.80f;
  const string PROGRAM_WINDOW_TITLE                            = "OpenGL Rendering Engine Version 14.0.0.0: ";
  const string ALL_TESTS_WINDOWS_TITLE[7]                      = { "Cube Capping (Test 1)", "Super Quadrics (Test 2)", "Model Loader (Test 3)",
                                                                   "Particle Rendering Super Quadrics (Test 4)", "Particle Rendering Model Loader (Test 5)",
                                                                   "Model Loader Depth-Of-Field (Test 6)", "Model Loader OIT (Test 7)"
                                                                 };
  enum AllTests : size_t { CUBE_CAPPING_TEST                      = 1, SUPER_QUADRICS_TEST                  = 2, MODEL_LOADER_TEST = 3,
                           PARTICLE_RENDERING_SUPER_QUADRICS_TEST = 4, PARTICLE_RENDERING_MODEL_LOADER_TEST = 5,
                           MODEL_LOADER_DEPTH_OF_FIELD_TEST       = 6, MODEL_LOADER_OIT_TEST                = 7
                         };
  unique_ptr<TestGLUTInterface> test(nullptr);

  /*
  *  Displays the program usage information.
  */
  inline void giveUsage(const char* cmd, int exitCode = EXIT_FAILURE)
  {
    cout << "\nUsage: " << cmd << " [-h][-help][-f][-test #][-texture #][-model #]\n";
#ifdef GPU_FRAMEWORK_DEBUG
    cout <<                        "[-disableMultiSample][-useModelLoaderDescriptor][-glDebug]\n";
#else
    cout <<                        "[-disableMultiSample][-useModelLoaderDescriptor]\n";
#endif // GPU_FRAMEWORK_DEBUG
    cout << "--------------------------------------------------------------------------\n";
    cout << "   -h or -help                    this help text\n";
    cout << "   -f                             full screen mode\n";
    cout << "   -test #                        start with test  #\n";
    cout << "   -texture #                     use texture file #\n";
    cout << "   -model #                       use model file   #\n";
    cout << "   -disableMultiSample            disable MultiSample (FSAA) mode\n";
    cout << "   -useModelLoaderDescriptor      use model loader descriptor file\n";
#ifdef GPU_FRAMEWORK_DEBUG
    cout << "   -glDebug                       use GL_DEBUG_OUTPUT reporting functionality\n";
#endif // GPU_FRAMEWORK_DEBUG

    exit(exitCode);
  }

  /*
  *  The outOfMemory message used from the new_handler of the TestGLUTInterface object instantiation.
  */
  inline void outOfMemory()
  {
    DebugConsole_consoleOutLine("Unable to satisfy request for memory for TestGLUTInterface!");
    terminate();
  }

  /*
  *  Uses a Factory Design Pattern for the various tests by up-casting to their TestIterface base (super) class.
  */
  inline unique_ptr<TestGLUTInterface> factoryDesignPatternForAllTests(AllTests currentTest, int screenWidth, int screenHeight, const string& textureFileName, const string& modelFileName, const string& modelLoaderDescriptorFileName, bool multisample)
  {
    // first set the outOfMemory new_handler for TestGLUTInterface
    TestGLUTInterface::set_new_handler(outOfMemory);
    switch (currentTest)
    {
      case CUBE_CAPPING_TEST:
        return make_unique<CubeCappingTest>(screenWidth, screenHeight, multisample);

      case SUPER_QUADRICS_TEST:
        return make_unique<SuperQuadricsTest>(screenWidth, screenHeight, textureFileName, multisample);

      case MODEL_LOADER_TEST:
        return make_unique<ModelLoaderTest>(screenWidth, screenHeight, textureFileName, modelFileName, modelLoaderDescriptorFileName, multisample);

      case PARTICLE_RENDERING_SUPER_QUADRICS_TEST:
        return make_unique<ParticleRenderingSuperQuadricsTest>(screenWidth, screenHeight, textureFileName, modelFileName, modelLoaderDescriptorFileName, multisample);

      case PARTICLE_RENDERING_MODEL_LOADER_TEST:
        return make_unique<ParticleRenderingModelLoaderTest>(screenWidth, screenHeight, textureFileName, modelFileName, modelLoaderDescriptorFileName, multisample);

      case MODEL_LOADER_DEPTH_OF_FIELD_TEST:
        return make_unique<ModelLoaderDepthOfFieldTest>(screenWidth, screenHeight, textureFileName, modelFileName, modelLoaderDescriptorFileName, multisample);

      case MODEL_LOADER_OIT_TEST:
        return make_unique<ModelLoaderOITTest>(screenWidth, screenHeight, textureFileName, modelFileName, modelLoaderDescriptorFileName, multisample);

      default:
        return make_unique<CubeCappingTest>(screenWidth, screenHeight, multisample);
    }
  }
}

// Callback function for glut functionality
extern "C" void renderSceneWrapper()
{
  test->renderScene();
}

// Callback function for glut functionality
extern "C" void changeSizeWrapper(int w, int h)
{
  test->changeSize(w, h);
}

// Callback function for glut functionality
extern "C" void keyboardWrapper(unsigned char key, int x, int y)
{
  test->keyboard(key, x, y);
}

// Callback function for glut functionality
extern "C" void specialKeysKeyboardWrapper(int key, int x, int y)
{
  test->specialKeysKeyboard(key, x, y);
}

// Callback function for glut functionality
extern "C" void mousePressWrapper(int button, int state, int x, int y)
{
  test->mouse(button, state, x, y);
}

// Callback function for glut functionality
extern "C" void mouseMotionWrapper(int x, int y)
{
  test->mouseMotion(x, y);
}

// Callback function for glut functionality
extern "C" void closeFuncWrapper()
{
  test->closeFunc();
}

#ifdef GPU_FRAMEWORK_DEBUG
// Translates and formats the OpenGL callback information to readable output
extern "C" void formatDebugOutput(char* outStr, size_t outStrSize, GLenum source, GLenum type, GLuint id, GLenum severity, const char* msg)
{
  array<char, 32> sourceStr = { { ' ' } };
  string sourceStrFormat = "UNDEFINED(0x%04X)";
  switch (source)
  {
    case GL_DEBUG_SOURCE_API_ARB:             sourceStrFormat = "API";             break;
    case GL_DEBUG_SOURCE_WINDOW_SYSTEM_ARB:   sourceStrFormat = "WINDOW_SYSTEM";   break;
    case GL_DEBUG_SOURCE_SHADER_COMPILER_ARB: sourceStrFormat = "SHADER_COMPILER"; break;
    case GL_DEBUG_SOURCE_THIRD_PARTY_ARB:     sourceStrFormat = "THIRD_PARTY";     break;
    case GL_DEBUG_SOURCE_APPLICATION_ARB:     sourceStrFormat = "APPLICATION";     break;
    case GL_DEBUG_SOURCE_OTHER_ARB:           sourceStrFormat = "OTHER";           break;
    default:                                  sourceStrFormat = "";                break;
  }
  snprintf(sourceStr.data(), 32, sourceStrFormat.c_str(), source);

  array<char, 32> typeStr = { { ' ' } };
  string typeStrFormat = "UNDEFINED(0x%04X)";
  switch (type)
  {
    case GL_DEBUG_TYPE_ERROR_ARB:               typeStrFormat = "ERROR";               break;
    case GL_DEBUG_TYPE_DEPRECATED_BEHAVIOR_ARB: typeStrFormat = "DEPRECATED_BEHAVIOR"; break;
    case GL_DEBUG_TYPE_UNDEFINED_BEHAVIOR_ARB:  typeStrFormat = "UNDEFINED_BEHAVIOR";  break;
    case GL_DEBUG_TYPE_PORTABILITY_ARB:         typeStrFormat = "PORTABILITY";         break;
    case GL_DEBUG_TYPE_PERFORMANCE_ARB:         typeStrFormat = "PERFORMANCE";         break;
    case GL_DEBUG_TYPE_OTHER_ARB:               typeStrFormat = "OTHER";               break;
    default:                                    typeStrFormat = "";                    break;
  }
  snprintf(typeStr.data(), 32, typeStrFormat.c_str(), type);

  array<char, 32> severityStr = { { ' ' } };
  string severityStrFormat = "UNDEFINED";
  switch (severity)
  {
    case GL_DEBUG_SEVERITY_HIGH_ARB:   severityStrFormat = "HIGH";   break;
    case GL_DEBUG_SEVERITY_MEDIUM_ARB: severityStrFormat = "MEDIUM"; break;
    case GL_DEBUG_SEVERITY_LOW_ARB:    severityStrFormat = "LOW";    break;
    default:                           severityStrFormat = "";       break;
  }
  snprintf(severityStr.data(), 32, severityStrFormat.c_str(), severity);

  // final string output
  snprintf(outStr, outStrSize, "OpenGL: %s [source=%s type=%s severity=%s id=%d]", msg, sourceStr.data(), typeStr.data(), severityStr.data(), id);
}

// Callback function for glDebugContext functionality
extern "C" void openglCallbackFunction(GLenum source, GLenum type, GLuint id, GLenum severity, GLsizei length, const GLchar* message, const GLvoid* userParam)
{
  array<char, 512> finalMessage = { { ' ' } };
  formatDebugOutput(finalMessage.data(), finalMessage.size(), source, type, id, severity, message);
  DebugConsole_consoleOutLine(finalMessage.data());
}
#endif // GPU_FRAMEWORK_DEBUG

int main(int argc, char* argv[])
{
  try
  {
#ifdef GPU_FRAMEWORK_DEBUG
    DebugConsole::setUseLogFile(true);
    DebugConsole::setLogFileName("GPUFrameworkMain.log");
#endif // GPU_FRAMEWORK_DEBUG

    ConfigFile configFile;

    // Parse the command line arguments
    AllTests currentTest                 = AllTests(configFile.getTest());
    string testTitle                     = ALL_TESTS_WINDOWS_TITLE[currentTest - 1];
    bool fullscreen                      = configFile.getFullScreen();
    bool multisample                     = configFile.getMultiSample();
#ifdef GPU_FRAMEWORK_DEBUG
    bool glDebug                         = configFile.getGLDebug();
#endif // GPU_FRAMEWORK_DEBUG
    string textureFileName               = configFile.getTextureFileName();
    string modelFileName                 = configFile.getModelFileName();
    string modelLoaderDescriptorFileName = configFile.getUseModelLoaderDescriptor() ? configFile.getModelLoaderDescriptorFileName() : "";
    for (size_t i = 1; i < argc; ++i)
    {
      const string currentOption = string(argv[i]);
      if ((currentOption == "-h") || (currentOption == "-help"))
      {
        giveUsage(argv[0], EXIT_SUCCESS);
      }
      else if (currentOption == "-f")
      {
        fullscreen = true;
        DebugConsole_consoleOutLine("Command-line argument fullscreen: ", StringAuxiliaryFunctions::toString<bool>(fullscreen));
      }
      else if (currentOption == "-disableMultiSample")
      {
        multisample = false;
        DebugConsole_consoleOutLine("Command-line argument disableMultiSample: ", StringAuxiliaryFunctions::toString<bool>(multisample));
      }
#ifdef GPU_FRAMEWORK_DEBUG
      else if (currentOption == "-glDebug")
      {
        glDebug = true;
        DebugConsole_consoleOutLine("Command-line argument glDebug: ", StringAuxiliaryFunctions::toString<bool>(glDebug));
      }
#endif // GPU_FRAMEWORK_DEBUG
      else if (currentOption == "-test")
      {
        if ((i + 1) >= argc) giveUsage(argv[0]);
        currentTest = AllTests(StringAuxiliaryFunctions::fromString<size_t>(argv[i + 1]));
        testTitle   = ALL_TESTS_WINDOWS_TITLE[currentTest - 1];
        ++i; // increment counter to skip the i + 1 option
        DebugConsole_consoleOutLine("Command-line argument test: ", testTitle);
      }
      else if (currentOption == "-texture")
      {
        if ((i + 1) >= argc) giveUsage(argv[0]);
        textureFileName = string(argv[i + 1]);
        ++i; // increment counter to skip the i + 1 option
        DebugConsole_consoleOutLine("Command-line argument textureFileName: ", textureFileName);
      }
      else if (currentOption == "-model")
      {
        if ((i + 1) >= argc) giveUsage(argv[0]);
        modelFileName = string(argv[i + 1]);
        ++i; // increment counter to skip the i + 1 option
        DebugConsole_consoleOutLine("Command-line argument modelFileName: ", modelFileName);
      }
      else if (currentOption == "-useModelLoaderDescriptor")
      {
        modelLoaderDescriptorFileName = configFile.getModelLoaderDescriptorFileName();
        DebugConsole_consoleOutLine("Command-line argument useModelLoaderDescriptor: ", modelLoaderDescriptorFileName);
      }
      else
      {
        giveUsage(argv[0]);
      }
    }

    glutInit(&argc, argv);
    const int screenWidth  = int(GLUT_WINDOW_TO_SCREEN_SIZE_RATIO * glutGet(GLUT_SCREEN_WIDTH));
    const int screenHeight = int(GLUT_WINDOW_TO_SCREEN_SIZE_RATIO * glutGet(GLUT_SCREEN_HEIGHT));
    glutSetOption(GLUT_ACTION_ON_WINDOW_CLOSE, GLUT_ACTION_CONTINUE_EXECUTION);
    unsigned int displayMode = GLUT_DOUBLE | GLUT_RGBA | GLUT_ALPHA | GLUT_DEPTH | GLUT_STENCIL | GLUT_ACCUM;
    if (multisample)
    {
      displayMode |= GLUT_MULTISAMPLE;
    }
    if (currentTest == MODEL_LOADER_OIT_TEST)
    {
      glutInitContextVersion(4, 3); // minimum support needed for OIT is GL 4.3
    }
    else
    {
      glutInitContextVersion(2, 1); // minimum support needed for rest of tests is GL 2.1 (& GL 3.3 for GBuffer effects & GS mode particles managed internally in relevant tests)
    }
    glutInitContextProfile(GLUT_COMPATIBILITY_PROFILE);
    glutInitContextFlags (GLUT_FORWARD_COMPATIBLE | GLUT_DEBUG);
#ifdef GPU_FRAMEWORK_DEBUG
    glutInitContextFlags(glDebug ? GLUT_FORWARD_COMPATIBLE | GLUT_DEBUG : GLUT_FORWARD_COMPATIBLE);
#else
    glutInitContextFlags(GLUT_FORWARD_COMPATIBLE);
#endif // GPU_FRAMEWORK_DEBUG
    glutInitDisplayMode(displayMode);
    glutInitWindowPosition((glutGet(GLUT_SCREEN_WIDTH) - screenWidth) / 2, (glutGet(GLUT_SCREEN_HEIGHT) - screenHeight) / 2);
    glutInitWindowSize(screenWidth, screenHeight);
    glutCreateWindow(string(PROGRAM_WINDOW_TITLE + testTitle).c_str());
    if (fullscreen)
    {
      glutFullScreen();
    }

    glewExperimental = GL_TRUE;
    const GLenum glewInitResult  = glewInit();
    if (GLEW_OK != glewInitResult)
    {
      // GLEW failed!
      DebugConsole_consoleOutLine("Error: ", glewGetString(glewInitResult));

      return EXIT_FAILURE;
    }

#ifdef GPU_FRAMEWORK_DEBUG
    if (glDebug)
    {
      if (glewIsSupported("GL_ARB_debug_output"))
      {
        DebugConsole_consoleOutLine("\nRegistering OpenGL debug callback");
        glEnable(GL_DEBUG_OUTPUT_SYNCHRONOUS);
        // control which messages are displayed
        glDebugMessageControl(GL_DONT_CARE, GL_DONT_CARE, GL_DONT_CARE, 0, nullptr, GL_FALSE); // first disable everything
        // then enable messages of severity high
        glDebugMessageControl(GL_DONT_CARE, // which source to define
                              GL_DONT_CARE, // type of message
                              GL_DONT_CARE, // severity level
                              0, nullptr,   // check the spec
                              GL_TRUE);     // enable or disable everything specified above
        // set callback function for debug info
        glDebugMessageCallback((GLDEBUGPROC)&openglCallbackFunction, nullptr);
      }
      else
      {
        DebugConsole_consoleOutLine("GL_ARB_debug_output not available");
      }
    }
#endif // GPU_FRAMEWORK_DEBUG

    DebugConsole_consoleOutLine("\nStarting now with options:");
    DebugConsole_consoleOutLine("fullscreen:      ", StringAuxiliaryFunctions::toString<bool>(fullscreen));
    DebugConsole_consoleOutLine("multisample:     ", StringAuxiliaryFunctions::toString<bool>(multisample));
#ifdef GPU_FRAMEWORK_DEBUG
    DebugConsole_consoleOutLine("glDebug:         ", StringAuxiliaryFunctions::toString<bool>(glDebug));
#endif // GPU_FRAMEWORK_DEBUG
    DebugConsole_consoleOutLine("currentTest:     ", currentTest);
    DebugConsole_consoleOutLine("textureFileName: ", textureFileName);
    DebugConsole_consoleOutLine("modelFileName:   ", modelFileName);
    DebugConsole_consoleOutLine("modelLoaderDescriptorFileName: ", modelLoaderDescriptorFileName);
    DebugConsole_consoleOutLine();

#ifdef _WIN32
    Utils::SIMDVectorizations::reportCPUCapabilities();
#endif // _WIN32

    test = factoryDesignPatternForAllTests(currentTest, screenWidth, screenHeight, textureFileName, modelFileName, modelLoaderDescriptorFileName, multisample); // copy elision exploited here to not need an explicit move
    glutKeyboardFunc(keyboardWrapper);
    glutSpecialFunc(specialKeysKeyboardWrapper);
    glutMouseFunc(mousePressWrapper);
    glutMotionFunc(mouseMotionWrapper);
    glutReshapeFunc(changeSizeWrapper);
    glutDisplayFunc(renderSceneWrapper);
    glutIdleFunc(renderSceneWrapper);
    glutCloseFunc(closeFuncWrapper);
    glutMainLoop();

    DebugConsole_consoleOutLine("\nNow exiting the GPU Framework & Rendering Engine Tests.\n");

    return EXIT_SUCCESS;
  }
  catch (...)
  {
    DebugConsole_consoleOutLine("\nmain() reported exception!\n");

    return EXIT_FAILURE;
  }
}