/*

Copyright (c) 2009-2018, Thanos Theo. All rights reserved.
Released Under a Simplified BSD (FreeBSD) License
for academic, personal & non-commercial use.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the author and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.

A Commercial License is also available for commercial use with
special restrictions and obligations at a one-off fee. See links at:
1. http://www.dotredconsultancy.com/openglrenderingenginetoolrelease.php
2. http://www.dotredconsultancy.com/openglrenderingenginetoolsourcecodelicence.php
Please contact Thanos Theo (thanos.theo@dotredconsultancy.com) for more information.

*/

#pragma once

#ifndef INTENTIVE_CACHE_CLEANING_H
#define INTENTIVE_CACHE_CLEANING_H

#include "CUDAConstants.h"
#include "CUDADeviceScopeGuard.h"
#include "CUDAMemoryHandlers.h"
#include "CUDAParallelFor.h"
#include "CUDAStreamsHandler.h"
#include "CUDAUtilityFunctions.h"
#include <cuda_runtime_api.h>
#include <cstdint>

/** @brief namespace UtilsCUDA for encapsulating all the CUDA related code compiled by the NVCC compiler.
*
* @author Thanos Theo, 2018
* @version 14.0.0.0
*/
namespace UtilsCUDA
{
  /** The IntentiveCacheCleaning class intent to clean GPU L2 cache by performing synthetic operation on GPU global memory.
  *
  * @author Amir Shahvarani, 2019
  * @version 14.0.0.0
  */
  class IntentiveCacheCleaning final
  {
  public:

    IntentiveCacheCleaning(std::size_t dataSize = 1024 * 1024 * 1024, const cudaStream_t& cudaStream = nullptr, int device = NO_DEVICE_CHANGE)
      : dataSize_(dataSize)
      , cacheData_(dataSize, device)
      , cudaStream_(cudaStream)
      , device_(device)
    {
      cacheData_.memsetAsync(0, cudaStream);
    }

    ~IntentiveCacheCleaning()
    {
      cacheData_.reset();
    }

    void cleanCache()
    {
      CUDADeviceScopeGuard deviceScopeGuard(device_); // push given device on stack (previously set device will be restored at end of scope using RAII)
      CUDAParallelFor::launchCUDAParallelForInStream(dataSize_, 0, cudaStream_, [] __device__ (std::size_t index, std::uint32_t* __restrict data)
      {
        ++data[index];
      }, cacheData_.device());
      CUDAError_checkCUDAError(cudaStreamSynchronize(cudaStream_));
    }

private:

    std::size_t dataSize_ = 1024 * 1024 * 1024;
    DeviceMemory<std::uint32_t> cacheData_;
    const cudaStream_t cudaStream_{};
    int device_ = NO_DEVICE_CHANGE;
  };
} // namespace UtilsCUDA

#endif // INTENTIVE_CACHE_CLEANING_H